unit JAWSImplementation;

interface

{ DONE -oJeremy Merrill -c508 :
  Add something that prevents overwriting of the script files if another
  app is running that's using the JAWS DLL }
{ TODO -oJeremy Merrill -c508 : Add check in here to look at script version in JSS file }
{ DONE -oJeremy Merrill -c508 :
  Replace registry communication with multiple windows - save strings in the window titles
  Use EnumerateChildWindows jaws script function in place of the FindWindow function
  that's being used right now.- EnumerateChildWindows with a window handle of 0
  enumerates all windows on the desktop.  Will have to use the first part of the window
  title as an ID, and the last part as the string values.  Will need to check for a maximum
  string lenght, probably have to use multiple windows for long text.
  Will also beed to have a global window shared by muiltiple instances of the JAWS.SR DLL. }
{ DONE -oJeremy Merrill -c508 :
  Need to add version checking to TVA508AccessibilityManager component
  and JAWS.DLL.  Warning needs to display just like JAWS.DLL and JAWS. }
uses SysUtils, Windows, Classes, Registry, StrUtils, Forms, Dialogs,
  ExtCtrls, VAUtils, DateUtils, PSApi, IniFiles, ActiveX,
  SHFolder, ShellAPI, VA508AccessibilityConst;

{$I 'VA508ScreenReaderDLLStandard.inc'}
{ DONE -oJeremy Merrill -c508 :Figure out why Delphi IDE is loading the DLL when JAWS is running  -
  probably has something to do with the VA508 package being installed -
  need to test for csDesigning some place that we're not testing for (maybe?) }

exports Initialize, ShutDown, RegisterCustomBehavior, ComponentData, SpeakText,
  IsRunning, ConfigChangePending;

implementation

uses fVA508HiddenJawsMainWindow, FSAPILib_TLB, ComObj;

const
  // JAWS_REQUIRED_VERSION     = '7.10.500'; in VA508AccessibilityConst unit
  JAWS_COM_OBJECT_VERSION = '8.0.2173';

  VA508_REG_PARAM_KEY = 'Software\Vista\508\JAWS';

  VA508_REG_COMPONENT_CAPTION = 'Caption';
  VA508_REG_COMPONENT_VALUE = 'Value';
  VA508_REG_COMPONENT_CONTROL_TYPE = 'ControlType';
  VA508_REG_COMPONENT_STATE = 'State';
  VA508_REG_COMPONENT_INSTRUCTIONS = 'Instructions';
  VA508_REG_COMPONENT_ITEM_INSTRUCTIONS = 'ItemInstructions';
  VA508_REG_COMPONENT_DATA_STATUS = 'DataStatus';

  VA508_ERRORS_SHOWN_STATE = 'ErrorsShown';

  RELOAD_CONFIG_SCRIPT = 'VA508Reload';

  SLASH = '\';
  { TODO -oJeremy Merrill -c508 :
    Change APP_DATA so that "application data" isn't used - Windows Vista
    doesn't use this value - get data from Windows API call }
  APP_DATA = SLASH + 'application data' + SLASH;
  JAWS_COMMON_SCRIPT_PATH_TEXT = 'freedom scientific\jaws\';
  JAWS_COMMON_SCRIPT_PATH_TEXT_LEN = length(JAWS_COMMON_SCRIPT_PATH_TEXT);

  // VHAISPBELLC 8/20/2013
  JAWS_REGROOT = 'SOFTWARE\Freedom Scientific\JAWS';
  JAWS_SCRIPTDIR = 'SETTINGS\enu';
  JAWS_INSTALL_DIRECTORY_VAR = 'Target';
  JAWS_SHARED_DIR = 'Shared\';
  KEY_WOW64_64KEY = $0100;

type
  TCompareType = (jcPrior, jcINI, jcLineItems, jcVersion, jcScriptMerge);

  TFileInfo = record
    AppFile: boolean;
    Ext: string;
    CompareType: TCompareType;
    Required: boolean;
    Compile: boolean;
  end;

const
  JAWS_SCRIPT_NAME = 'VA508JAWS';

  JAWS_SCRIPT_VERSION = 'VA508_Script_Version';
  CompiledScriptFileExtension = '.JSB';
  ScriptFileExtension = '.JSS';
  ScriptDocExtension = '.JSD';
  ConfigFileExtension = '.JCF';
  KeyMapExtension = '.JKM';
  DictionaryFileExtension = '.JDF';

  FileInfo: array [1 .. 6] of TFileInfo = ((AppFile: FALSE;
    Ext: ScriptFileExtension; CompareType: jcVersion; Required: TRUE;
    Compile: TRUE), (AppFile: FALSE; Ext: ScriptDocExtension;
    CompareType: jcPrior; Required: TRUE; Compile: FALSE), (AppFile: TRUE;
    Ext: ScriptFileExtension; CompareType: jcScriptMerge; Required: TRUE;
    Compile: TRUE), (AppFile: TRUE; Ext: ConfigFileExtension;
    CompareType: jcINI; Required: TRUE; Compile: FALSE), (AppFile: TRUE;
    Ext: DictionaryFileExtension; CompareType: jcLineItems; Required: FALSE;
    Compile: FALSE), (AppFile: TRUE; Ext: KeyMapExtension; CompareType: jcINI;
    Required: FALSE; Compile: FALSE));

  JAWS_VERSION_ERROR = ERROR_INTRO +
    'The Accessibility Framework can only communicate with JAWS ' +
    JAWS_REQUIRED_VERSION + CRLF +
    'or later versions.  Please update your version of JAWS to a minimum of' +
    CRLF + JAWS_REQUIRED_VERSION +
    ', or preferably the most recent release, to allow the Accessibility' + CRLF
    + 'Framework to communicate with JAWS.  If you are getting this message' +
    CRLF + 'and you already have a compatible version of JAWS, please contact your'
    + CRLF + 'system administrator, and request that they run, with administrator rights,'
    + CRLF + 'the JAWSUpdate application located in the \Program Files\VistA\' +
    CRLF + 'Common Files directory. JAWSUpdate is not required for JAWS' + CRLF
    + 'versions ' + JAWS_COM_OBJECT_VERSION + ' and above.' + CRLF;

  JAWS_FILE_ERROR = ERROR_INTRO +
    'The JAWS interface with the Accessibility Framework requires the ability' +
    CRLF + 'to write files to the hard disk, but the following error is occurring trying to'
    + CRLF + 'write to the disk:' + CRLF + '%s' + CRLF +
    'Please contact your system administrator in order to ensure that your ' +
    CRLF + 'security privileges allow you to write files to the hard disk.' +
    CRLF + 'If you are sure you have these privileges, your hard disk may be full.  Until'
    + CRLF + 'this problem is resolved, the Accessibility Framework will not be able to'
    + CRLF + 'communicate with JAWS.';

  JAWS_USER_MISSMATCH_ERROR = ERROR_INTRO +
    'An error has been detected in the state of JAWS that will not allow the' +
    CRLF + 'Accessibility Framework to communicate with JAWS until JAWS is shut'
    + CRLF + 'down and restarted.  Please restart JAWS at this time.';

  DLL_VERSION_ERROR = ERROR_INTRO +
    'The Accessibility Framework is at version %s, but the required JAWS' + CRLF
    + 'support files are only at version %s.  The new support files should have'
    + CRLF + 'been released with the latest version of the software you are currently'
    + CRLF + 'running.  The Accessibility Framework will not be able to communicate'
    + CRLF + 'with JAWS until these support files are installed.  Please contact your'
    + CRLF + 'system administrator for assistance.';

  JAWS_ERROR_VERSION = 1;
  JAWS_ERROR_FILE_IO = 2;
  JAWS_ERROR_USER_PROBLEM = 3;
  DLL_ERROR_VERSION = 4;

  JAWS_ERROR_COUNT = 4;

  JAWS_RELOAD_DELAY = 500;

var
  JAWSErrorMessage: array [1 .. JAWS_ERROR_COUNT] of string = (
    JAWS_VERSION_ERROR,
    JAWS_FILE_ERROR,
    JAWS_USER_MISSMATCH_ERROR,
    DLL_VERSION_ERROR
  );

  JAWSErrorsShown: array [1 .. JAWS_ERROR_COUNT] of boolean = (
    FALSE,
    FALSE,
    FALSE,
    FALSE
  );

type
  TJAWSSayString = function(StringToSpeak: PChar; Interrupt: BOOL)
    : BOOL; stdcall;
  TJAWSRunScript = function(ScriptName: PChar): BOOL; stdcall;

  TStartupID = record
    Handle: HWND;
    InstanceID: Integer;
    MsgID: Integer;
  end;

  // VHAISPBELLC 8/27/2013
  TJawsRecord = record
    Version: Double;
    Compiler: string;
    DefaultScriptDir: String;
    UserStriptDir: String;
    FDictionaryFileName: string;
    FConfigFile: string;
    FKeyMapFile: string;
    FKeyMapINIFile: TINIFile;
    FKeyMapINIFileModified: boolean;
    FAssignedKeys: TStringList;
    FConfigINIFile: TINIFile;
    FConfigINIFileModified: boolean;
    FDictionaryFile: TStringList;
    FDictionaryFileModified: boolean;
  end;

  TJAWSManager = class
  strict private
    FRequiredFilesFound: boolean;
    FMainForm: TfrmVA508HiddenJawsMainWindow;
    FWasShutdown: boolean;
    FJAWSFileError: string;
    FMasterApp: string;
    FRootScriptFileName: string;
    FRootScriptAppFileName: string;
    JAWSAPI: IJawsApi;
  private
    procedure ShutDown;
    procedure MakeFileWritable(FileName: string);
    procedure LaunchMasterApplication;
    procedure KillINIFiles(Sender: TObject);
    procedure ReloadConfiguration;
  public
    constructor Create;
    destructor Destroy; override;
    class procedure ShowError(ErrorNumber: Integer); overload;
    class procedure ShowError(ErrorNumber: Integer;
      data: array of const); overload;
    class function GetPathFromJAWS(PathID: Integer;
      DoLowerCase: boolean = TRUE): string;
    class function GetJAWSWindow: HWND;
    class function IsRunning(HighVersion, LowVersion: Word): BOOL;
    function Initialize(ComponentCallBackProc: TComponentDataRequestProc): BOOL;
    procedure SendComponentData(WindowHandle: HWND; DataStatus: LongInt;
      Caption, Value, data, ControlType, State, Instructions,
      ItemInstructions: PChar);
    procedure SpeakText(Text: PChar);
    procedure RegisterCustomBehavior(Before, After: string; Action: Integer);
    class function JAWSVersionOK: boolean;
    class function JAWSTalking2CurrentUser: boolean;
    function FileErrorExists: boolean;
    property RequiredFilesFound: boolean read FRequiredFilesFound;
    property MainForm: TfrmVA508HiddenJawsMainWindow read FMainForm;

  end;

var
  JAWSManager: TJAWSManager = nil;
  DLLMessageID: UINT = 0;
  JawsRecord: array of TJawsRecord;

procedure AddToLog(TextToAdd: string; Level: Integer = 1);
VAR
  OurLogFile, LocalOnly, AppDir: string;
  FS: TFileStream;
  Flags: Word;
  logFile: File;
  I: Integer;

  // Finds the users special directory
  function LocalAppDataPath: string;
  const
    SHGFP_TYPE_CURRENT = 0;
  var
    path: array [0 .. MaxChar] of char;
    TmpPath: PChar;
    PathLen: Integer;
  begin
    SHGetFolderPath(0, CSIDL_LOCAL_APPDATA, 0, SHGFP_TYPE_CURRENT, @path[0]);
    Result := StrPas(path);
  end;

begin
{$IFDEF LogMode}
  OurLogFile := LocalAppDataPath;
  if (copy(OurLogFile, length(OurLogFile), 1) <> '\') then
    OurLogFile := OurLogFile + '\';

  LocalOnly := OurLogFile;

  // Now set the application level
  OurLogFile := OurLogFile + Application.title;
  if (copy(OurLogFile, length(OurLogFile), 1) <> '\') then
    OurLogFile := OurLogFile + '\';
  AppDir := OurLogFile;

  // try to create or use base direcrtory
  if not DirectoryExists(AppDir) then
    if not ForceDirectories(AppDir) then
      OurLogFile := LocalOnly;

  OurLogFile := OurLogFile + 'JAWSLog.txt';

  If FileExists(OurLogFile) then
    Flags := fmOpenReadWrite
  else
    Flags := fmCreate;

  for I := 2 to Level do
    TextToAdd := ' ' + TextToAdd;

  TextToAdd := #13#10 + TextToAdd;
  FS := TFileStream.Create(OurLogFile, Flags);
  try
    FS.Position := FS.Size;
    FS.Write(TextToAdd[1], length(TextToAdd) * SizeOf(char));
  finally
    FS.Free;
  end;
{$ENDIF}
end;

procedure EnsureManager;
begin
  if not assigned(JAWSManager) then
    JAWSManager := TJAWSManager.Create;
end;

// Checks to see if the screen reader is currently running
function IsRunning(HighVersion, LowVersion: Word): BOOL; stdcall;
begin
  EnsureManager; // need to preload the directories
  Result := TJAWSManager.IsRunning(HighVersion, LowVersion);
  AddToLog('IsRunning: ' + BoolToStr(Result), 3);
end;

// Executed after IsRunning returns TRUE, when the DLL is accepted as the screen reader of choice
function Initialize(ComponentCallBackProc: TComponentDataRequestProc)
  : BOOL; stdcall;
begin
  EnsureManager;
  Result := JAWSManager.Initialize(ComponentCallBackProc);
end;

// Executed when the DLL is unloaded or screen reader is no longer needed
procedure ShutDown; stdcall;
begin
  if assigned(JAWSManager) then
  begin
    JAWSManager.ShutDown;
    FreeAndNil(JAWSManager);
  end;
end;

function ConfigChangePending: boolean; stdcall;
begin
  Result := FALSE;
  if assigned(JAWSManager) and assigned(JAWSManager.MainForm) and
    (JAWSManager.MainForm.ConfigChangePending) then
    Result := TRUE;
end;

// Returns Component Data as requested by the screen reader
procedure ComponentData(WindowHandle: HWND; DataStatus: LongInt = DATA_NONE;
  Caption: PChar = nil; Value: PChar = nil; data: PChar = nil;
  ControlType: PChar = nil; State: PChar = nil; Instructions: PChar = nil;
  ItemInstructions: PChar = nil); stdcall;
begin
  EnsureManager;
  JAWSManager.SendComponentData(WindowHandle, DataStatus, Caption, Value, data,
    ControlType, State, Instructions, ItemInstructions);
end;

// Instructs the Screen Reader to say the specified text
procedure SpeakText(Text: PChar); stdcall;
begin
  EnsureManager;
  JAWSManager.SpeakText(Text);
end;

procedure RegisterCustomBehavior(BehaviorType: Integer; Before, After: PChar);
begin
  EnsureManager;
  JAWSManager.RegisterCustomBehavior(Before, After, BehaviorType);
end;

{ TJAWSManager }

const
{$WARNINGS OFF} // Don't care about platform specific warning
  NON_WRITABLE_FILE_ATTRIB = faReadOnly or faHidden;
{$WARNINGS ON}
  WRITABLE_FILE_ATTRIB = faAnyFile and (not NON_WRITABLE_FILE_ATTRIB);

procedure TJAWSManager.MakeFileWritable(FileName: string);
var
  Attrib: Integer;
begin
{$WARNINGS OFF} // Don't care about platform specific warning
  Attrib := FileGetAttr(FileName);
{$WARNINGS ON}
  if (Attrib and NON_WRITABLE_FILE_ATTRIB) <> 0 then
  begin
    Attrib := Attrib and WRITABLE_FILE_ATTRIB;
{$WARNINGS OFF} // Don't care about platform specific warning
    if FileSetAttr(FileName, Attrib) <> 0 then
{$WARNINGS ON}
      FJAWSFileError := 'Could not change read-only attribute of file "' +
        FileName + '"';
  end;
end;

var
  JAWSMsgID: UINT = 0;

const
  JAWS_MESSAGE_ID = 'JW_GET_FILE_PATH';
  // version is in directory after JAWS \Freedom Scientific\JAWS\*.*\...
  JAWS_PATH_ID_APPLICATION = 0;
  JAWS_PATH_ID_USER_SCRIPT_FILES = 1;
  JAWS_PATH_ID_JAWS_DEFAULT_SCRIPT_FILES = 2;
  // 0 = C:\Program Files\Freedom Scientific\JAWS\8.0\jfw.INI
  // 1 = D:\Documents and Settings\vhaislmerrij\Application Data\Freedom Scientific\JAWS\8.0\USER.INI
  // 2 = D:\Documents and Settings\All Users\Application Data\Freedom Scientific\JAWS\8.0\Settings\enu\DEFAULT.SBL

class function TJAWSManager.GetPathFromJAWS(PathID: Integer;
  DoLowerCase: boolean = TRUE): string;
var
  atm: ATOM;
  len: Integer;
  path: string;
  JAWSWindow: HWND;
begin
  JAWSWindow := GetJAWSWindow;
  if JAWSMsgID = 0 then
    JAWSMsgID := RegisterWindowMessage(JAWS_MESSAGE_ID);
  Result := '';
  atm := SendMessage(JAWSWindow, JAWSMsgID, PathID, 0);
  if atm <> 0 then
  begin
    SetLength(path, MAX_PATH * 2);
    len := GlobalGetAtomName(atm, PChar(path), MAX_PATH * 2);
    GlobalDeleteAtom(atm);
    if len > 0 then
    begin
      SetLength(path, len);
      Result := ExtractFilePath(path);
      if DoLowerCase then
        Result := LowerCase(Result);
    end;
  end;
end;

constructor TJAWSManager.Create;
const
  COMPILER_FILENAME = 'scompile.exe';
  JAWS_APP_NAME = 'VA508APP';
  JAWSMasterApp = 'VA508JAWSDispatcher.exe';

  { -------------------------------------------------------------------------------
    Procedure:   LoadJawsDirectories
    Author:      VHAISPBELLC
    DateTime:    2013.08.27
    Arguments:   None
    Result:      None
    Description: Load the jaws directories
    ------------------------------------------------------------------------------- }
  procedure LoadJawsDirectories;
  var
    reg: TRegistry;
    keys: TStringList;
    idx, I: Integer;
    key, Dir, SubDir, Version: string;
  begin
    keys := TStringList.Create;
    try
      reg := TRegistry.Create(KEY_READ or KEY_WOW64_64KEY);
      try
        reg.RootKey := HKEY_LOCAL_MACHINE;
        reg.OpenKey(JAWS_REGROOT, FALSE);
        reg.GetKeyNames(keys);
        for I := 0 to keys.Count - 1 do
        begin
          Version := keys[I];
          key := JAWS_REGROOT + '\' + keys[I] + '\';
          reg.CloseKey;
          AddToLog(Version, 3);
          if reg.OpenKey(key, FALSE) then
          begin
            Dir := LowerCase(reg.ReadString(JAWS_INSTALL_DIRECTORY_VAR));
            Dir := AppendBackSlash(Dir) + COMPILER_FILENAME;
            AddToLog(Dir, 4);
            if FileExists(Dir) then
            begin
              SetLength(JawsRecord, length(JawsRecord) + 1);
              JawsRecord[high(JawsRecord)].Version :=
                StrTofloatDef(Version, -1);
              JawsRecord[high(JawsRecord)].Compiler := Dir;
              SubDir := Version + '\' + JAWS_SCRIPTDIR;
              Dir := GetSpecialFolderPath(CSIDL_COMMON_APPDATA) +
                JAWS_COMMON_SCRIPT_PATH_TEXT + AppendBackSlash(SubDir);
              JawsRecord[high(JawsRecord)].DefaultScriptDir := Dir;
              Dir := GetSpecialFolderPath(CSIDL_APPDATA) +
                JAWS_COMMON_SCRIPT_PATH_TEXT + AppendBackSlash(SubDir);
              JawsRecord[high(JawsRecord)].UserStriptDir := Dir;
            end;
          end;
        end;
      finally
        reg.Free;
      end;
    finally
      keys.Free;
    end;

  end;

  procedure FindJAWSRequiredFiles;
  var
    path: string;
    I: Integer;
    FileName: string;
    info: TFileInfo;

  begin
    SetLength(path, MAX_PATH);
    SetLength(path, GetModuleFileName(HInstance, PChar(path), length(path)));
    path := ExtractFilePath(path);
    path := AppendBackSlash(path);
    // look for the script files in the same directory as this DLL
    FRootScriptFileName := path + JAWS_SCRIPT_NAME;
    FRootScriptAppFileName := path + JAWS_APP_NAME;
    FRequiredFilesFound := TRUE;
    AddToLog('Checking files', 3);
    for I := low(FileInfo) to high(FileInfo) do
    begin
      info := FileInfo[I];
      if info.Required then
      begin
        if info.AppFile then
          FileName := FRootScriptAppFileName + info.Ext
        else
          FileName := FRootScriptFileName + info.Ext;
        if not FileExists(FileName) then
        begin
          FRequiredFilesFound := FALSE;
          AddToLog(FileName + ' does not exist', 4);
          break;
        end
        else
          AddToLog(FileName + ' does exist', 4);
      end;
    end;
    FMasterApp := path + JAWSMasterApp;
  end;

begin
  AddToLog(StringOfChar('*', 20) + FormatDateTime('mm/dd/yyyy hh:mm:ss', now) +
    StringOfChar('*', 20));
  SetLength(JawsRecord, 0);
  AddToLog('Loading Jaws files', 2);
  LoadJawsDirectories;
  if length(JawsRecord) > 0 then
    FindJAWSRequiredFiles;
end;

destructor TJAWSManager.Destroy;
begin
  SetLength(JawsRecord, 0);
  ShutDown;
  inherited;
end;

function TJAWSManager.FileErrorExists: boolean;
begin
  Result := (FJAWSFileError <> '');
end;

class function TJAWSManager.GetJAWSWindow: HWND;
const
  VISIBLE_WINDOW_CLASS: PChar = 'JFWUI2';
  VISIBLE_WINDOW_TITLE: PChar = 'JAWS';
  VISIBLE_WINDOW_TITLE2: PChar = 'Remote JAWS';

begin
  Result := FindWindow(VISIBLE_WINDOW_CLASS, VISIBLE_WINDOW_TITLE);
  if Result = 0 then
    Result := FindWindow(VISIBLE_WINDOW_CLASS, VISIBLE_WINDOW_TITLE2);
end;

function TJAWSManager.Initialize(ComponentCallBackProc
  : TComponentDataRequestProc): BOOL;
var
  DestPath: string;
  ScriptFileChanges: boolean;
  LastFileUpdated: boolean;
  CompileCommands: TStringList;
  AppScriptNeedsFunction: boolean;
  AppNeedsUseLine: boolean;
  AppUseLine: string;
  AppStartFunctionLine: Integer;
  ArryCnt: Integer;

  procedure EnsureWindow;
  begin
    if not assigned(FMainForm) then
      FMainForm := TfrmVA508HiddenJawsMainWindow.Create(nil);
    FMainForm.ComponentDataCallBackProc := ComponentCallBackProc;
    FMainForm.ConfigReloadProc := ReloadConfiguration;
    FMainForm.HandleNeeded;
    Application.ProcessMessages;
  end;

  function GetVersion(FileName: string): Integer;
  var
    list: TStringList;

    p, I: Integer;
    line: string;
    working: boolean;
  begin
    Result := 0;
    list := TStringList.Create;
    try
      list.LoadFromFile(FileName);
      I := 0;
      working := TRUE;
      while working and (I < list.Count) do
      begin
        line := list[I];
        p := pos('=', line);
        if p > 0 then
        begin
          if trim(copy(line, 1, p - 1)) = JAWS_SCRIPT_VERSION then
          begin
            line := trim(copy(line, p + 1, MaxInt));
            if copy(line, length(line), 1) = ',' then
              delete(line, length(line), 1);
            Result := StrToIntDef(line, 0);
            working := FALSE;
          end;
        end;
        inc(I);
      end;
    finally
      list.Free;
    end;
  end;

  function VersionDifferent(FromFile, ToFile: string): boolean;
  var
    FromVersion, ToVersion: Integer;
  begin
    FromVersion := GetVersion(FromFile);
    ToVersion := GetVersion(ToFile);
    Result := (FromVersion > ToVersion);
  end;

  function LineItemUpdateNeeded(FromFile, ToFile: string): boolean;
  var
    fromList, toList: TStringList;
    I, idx: Integer;
    line: string;
  begin
    Result := FALSE;
    fromList := TStringList.Create;
    toList := TStringList.Create;
    try
      fromList.LoadFromFile(FromFile);
      toList.LoadFromFile(ToFile);
      for I := 0 to fromList.Count - 1 do
      begin
        line := fromList[I];
        if trim(line) <> '' then
        begin
          idx := toList.IndexOf(line);
          if idx < 0 then
          begin
            Result := TRUE;
            break;
          end;
        end;
      end;
    finally
      toList.Free;
      fromList.Free;
    end;
  end;

  function INIUpdateNeeded(FromFile, ToFile: string): boolean;
  var
    FromINIFile, ToINIFile: TINIFile;
    Sections, Values: TStringList;
    I, j: Integer;
    section, key, val1, val2: string;
  begin
    Result := FALSE;
    Sections := TStringList.Create;
    Values := TStringList.Create;
    try
      FromINIFile := TINIFile.Create(FromFile);
      try
        ToINIFile := TINIFile.Create(ToFile);
        try
          FromINIFile.ReadSections(Sections);
          for I := 0 to Sections.Count - 1 do
          begin
            section := Sections[I];
            FromINIFile.ReadSectionValues(section, Values);
            for j := 0 to Values.Count - 1 do
            begin
              key := Values.Names[j];
              val1 := Values.ValueFromIndex[j];
              val2 := ToINIFile.ReadString(section, key, '');
              Result := (val1 <> val2);
              if Result then
                break;
            end;
            if Result then
              break;
          end;
        finally
          ToINIFile.Free;
        end;
      finally
        FromINIFile.Free;
      end;
    finally
      Sections.Free;
      Values.Free;
    end;
  end;

  function IsUseLine(data: string): boolean;
  var
    p: Integer;
  begin
    Result := (copy(data, 1, 4) = 'use ');
    if Result then
    begin
      Result := FALSE;
      p := pos('"', data);
      if p > 0 then
      begin
        p := posEX('"', data, p + 1);
        if p = length(data) then
          Result := TRUE;
      end;
    end;
  end;

  function IsFunctionLine(data: string): boolean;
  var
    p1, p2: Integer;
    line: string;
  begin
    Result := FALSE;
    line := data;
    p1 := pos(' ', line);
    if (p1 > 0) then
    begin
      if copy(line, 1, p1 - 1) = 'script' then
        Result := TRUE
      else
      begin
        p2 := posEX(' ', line, p1 + 1);
        if p2 > 0 then
        begin
          line := copy(line, p1 + 1, p2 - p1 - 1);
          if (line = 'function') then
            Result := TRUE;
        end;
      end;
    end;
  end;

  function CheckForUseLineAndFunction(FromFile, ToFile: string): boolean;
  var
    FromData: TStringList;
    ToData: TStringList;
    UseLine: string;
    I: Integer;
    line: string;

  begin
    Result := FALSE;
    FromData := TStringList.Create;
    ToData := TStringList.Create;
    try
      UseLine := '';
      AppUseLine := '';
      AppStartFunctionLine := -1;
      FromData.LoadFromFile(FromFile);
      for I := 0 to FromData.Count - 1 do
      begin
        line := LowerCase(trim(FromData[I]));
        if (UseLine = '') and IsUseLine(line) then
        begin
          UseLine := line;
          AppUseLine := FromData[I];
        end
        else if (AppStartFunctionLine < 0) and IsFunctionLine(line) then
          AppStartFunctionLine := I;
        if (UseLine <> '') and (AppStartFunctionLine >= 0) then
          break;
      end;
      if (UseLine = '') or (AppStartFunctionLine < 0) then
        exit;

      AppNeedsUseLine := TRUE;
      AppScriptNeedsFunction := TRUE;
      ToData.LoadFromFile(ToFile);
      for I := 0 to ToData.Count - 1 do
      begin
        line := LowerCase(trim(ToData[I]));
        if AppNeedsUseLine and IsUseLine(line) and (line = UseLine) then
          AppNeedsUseLine := FALSE
        else if AppScriptNeedsFunction and IsFunctionLine(line) then
          AppScriptNeedsFunction := FALSE;
        if (not AppNeedsUseLine) and (not AppScriptNeedsFunction) then
          break;
      end;
      if AppNeedsUseLine or AppScriptNeedsFunction then
        Result := TRUE;
    finally
      FromData.Free;
      ToData.Free;
    end;
  end;

  function UpdateNeeded(FromFile, ToFile: string;
    CompareType: TCompareType): boolean;
  begin
    Result := TRUE;
    try
      case CompareType of
        jcScriptMerge:
          Result := CheckForUseLineAndFunction(FromFile, ToFile);
        jcPrior:
          Result := LastFileUpdated;
        jcVersion:
          Result := VersionDifferent(FromFile, ToFile);
        jcINI:
          Result := INIUpdateNeeded(FromFile, ToFile);
        jcLineItems:
          Result := LineItemUpdateNeeded(FromFile, ToFile);
      end;
    except
      on E: Exception do
        FJAWSFileError := E.Message;
    end;
  end;

  procedure INIFileUpdate(FromFile, ToFile: String);
  var
    FromINIFile, ToINIFile: TINIFile;
    Sections, Values: TStringList;
    I, j: Integer;
    section, key, val1, val2: string;
    modified: boolean;
  begin
    modified := FALSE;
    Sections := TStringList.Create;
    Values := TStringList.Create;
    try
      FromINIFile := TINIFile.Create(FromFile);
      try
        ToINIFile := TINIFile.Create(ToFile);
        try
          FromINIFile.ReadSections(Sections);
          for I := 0 to Sections.Count - 1 do
          begin
            section := Sections[I];
            FromINIFile.ReadSectionValues(section, Values);
            for j := 0 to Values.Count - 1 do
            begin
              key := Values.Names[j];
              val1 := Values.ValueFromIndex[j];
              val2 := ToINIFile.ReadString(section, key, '');
              if (val1 <> val2) then
              begin
                ToINIFile.WriteString(section, key, val1);
                modified := TRUE;
              end;
            end;
          end;
        finally
          if modified then
            ToINIFile.UpdateFile();
          ToINIFile.Free;
        end;
      finally
        FromINIFile.Free;
      end;
    finally
      Sections.Free;
      Values.Free;
    end;
  end;

  procedure LineItemFileUpdate(FromFile, ToFile: string);
  var
    fromList, toList: TStringList;
    I, idx: Integer;
    line: string;
    modified: boolean;
  begin
    modified := FALSE;
    fromList := TStringList.Create;
    toList := TStringList.Create;
    try
      fromList.LoadFromFile(FromFile);
      toList.LoadFromFile(ToFile);
      for I := 0 to fromList.Count - 1 do
      begin
        line := fromList[I];
        if trim(line) <> '' then
        begin
          idx := toList.IndexOf(line);
          if idx < 0 then
          begin
            toList.Add(line);
            modified := TRUE;
          end;
        end;
      end;
    finally
      if modified then
        toList.SaveToFile(ToFile);
      toList.Free;
      fromList.Free;
    end;
  end;

  procedure DeleteCompiledFile(ToFile: string);
  var
    CompiledFile: string;
  begin
    CompiledFile := copy(ToFile, 1, length(ToFile) -
      length(ExtractFileExt(ToFile)));
    CompiledFile := CompiledFile + CompiledScriptFileExtension;
    if FileExists(CompiledFile) then
    begin
      MakeFileWritable(CompiledFile);
      DeleteFile(PChar(CompiledFile));
    end;
  end;

  function DoScriptMerge(FromFile, ToFile, ThisCompiler: string): boolean;
  var
    BackupFile: string;
    FromData: TStringList;
    ToData: TStringList;
    I, idx: Integer;
    ExitCode: Integer;
  begin
    Result := TRUE;
    BackupFile := ToFile + '.BACKUP';
    if FileExists(BackupFile) then
    begin
      MakeFileWritable(BackupFile);
      DeleteFile(PChar(BackupFile));
    end;
    DeleteCompiledFile(ToFile);
    CopyFile(PChar(ToFile), PChar(BackupFile), FALSE);
    MakeFileWritable(ToFile);
    FromData := TStringList.Create;
    ToData := TStringList.Create;
    try
      ToData.LoadFromFile(ToFile);
      if AppNeedsUseLine then
        ToData.Insert(0, AppUseLine);
      if AppScriptNeedsFunction then
      begin
        FromData.LoadFromFile(FromFile);
        ToData.Insert(1, '');
        idx := 2;
        for I := AppStartFunctionLine to FromData.Count - 1 do
        begin
          ToData.Insert(idx, FromData[I]);
          inc(idx);
        end;
        ToData.Insert(idx, '');
      end;
      if not assigned(JAWSAPI) then
        JAWSAPI := CoJawsApi.Create;
      ToData.SaveToFile(ToFile);
      ExitCode := ExecuteAndWait('"' + ThisCompiler + '"', '"' + ToFile + '"');
      JAWSAPI.StopSpeech;
      if ExitCode = 0 then // compile succeeded!
        ReloadConfiguration
      else
        Result := FALSE; // compile failed - just copy the new one
    finally
      FromData.Free;
      ToData.Free;
    end;
  end;

  procedure UpdateFile(FromFile, ToFile, ThisCompiler: string; info: TFileInfo);
  var
    DoCopy: boolean;
    error: boolean;
    CheckOverwrite: boolean;
  begin
    DoCopy := FALSE;
    if FileExists(ToFile) then
    begin
      MakeFileWritable(ToFile);
      CheckOverwrite := TRUE;
      try
        case info.CompareType of
          jcScriptMerge:
            if not DoScriptMerge(FromFile, ToFile, ThisCompiler) then
              DoCopy := TRUE;
          jcPrior, jcVersion:
            DoCopy := TRUE;
          jcINI:
            INIFileUpdate(FromFile, ToFile);
          jcLineItems:
            LineItemFileUpdate(FromFile, ToFile);
        end;
      except
        on E: Exception do
          FJAWSFileError := E.Message;
      end;
    end
    else
    begin
      CheckOverwrite := FALSE;
      DoCopy := TRUE;
    end;
    if DoCopy then
    begin
      error := FALSE;
      if not CopyFile(PChar(FromFile), PChar(ToFile), FALSE) then
        error := TRUE;
      if (not error) and (not FileExists(ToFile)) then
        error := TRUE;
      if (not error) and CheckOverwrite and (info.CompareType <> jcPrior) and
        UpdateNeeded(FromFile, ToFile, info.CompareType) then
        error := TRUE;
      if error and (not FileErrorExists) then
        FJAWSFileError := 'Error copying "' + FromFile + '" to' + CRLF + '"' +
          ToFile + '".';
      if (not error) and (info.Compile) then
      begin
        DeleteCompiledFile(ToFile);
        CompileCommands.Add('"' + ToFile + '"');
      end;
    end;
  end;

  procedure EnsureJAWSScriptsAreUpToDate(var ThisJawsRec: TJawsRecord);
  var
    DestFile, FromFile, ToFile, AppName, Ext: string;
    idx1, idx2, I: Integer;
    DoUpdate: boolean;
    info: TFileInfo;

  begin
    AppName := ExtractFileName(ParamStr(0));
    Ext := ExtractFileExt(AppName);
    AppName := LeftStr(AppName, length(AppName) - length(Ext));
    DestPath := '';
    idx1 := pos(JAWS_COMMON_SCRIPT_PATH_TEXT, ThisJawsRec.UserStriptDir);
    idx2 := pos(JAWS_COMMON_SCRIPT_PATH_TEXT, ThisJawsRec.DefaultScriptDir);
    if (idx1 > 0) and (idx2 > 0) then
    begin
      DestPath := copy(ThisJawsRec.UserStriptDir, 1, idx1 - 1) +
        copy(ThisJawsRec.DefaultScriptDir, idx2, MaxInt);
      DestFile := DestPath + AppName;
      ThisJawsRec.FDictionaryFileName := DestFile + DictionaryFileExtension;
      ThisJawsRec.FConfigFile := DestFile + ConfigFileExtension;
      ThisJawsRec.FKeyMapFile := DestFile + KeyMapExtension;
      LastFileUpdated := FALSE;
      for I := low(FileInfo) to high(FileInfo) do
      begin
        info := FileInfo[I];
        if info.AppFile then
        begin
          FromFile := FRootScriptAppFileName + info.Ext;
          ToFile := DestFile + info.Ext;
        end
        else
        begin
          FromFile := FRootScriptFileName + info.Ext;
          ToFile := DestPath + JAWS_SCRIPT_NAME + info.Ext;
        end;
        if not FileExists(FromFile) then
          continue;
        if FileExists(ToFile) then
        begin
          DoUpdate := UpdateNeeded(FromFile, ToFile, info.CompareType);
          if DoUpdate then
            MakeFileWritable(ToFile);
        end
        else
          DoUpdate := TRUE;
        LastFileUpdated := DoUpdate;
        if DoUpdate and (not FileErrorExists) then
        begin
          UpdateFile(FromFile, ToFile, ThisJawsRec.Compiler, info);
          ScriptFileChanges := TRUE;
        end;
        if FileErrorExists then
          break;
      end;
    end
    else
      FJAWSFileError := 'Unknown File Error';
    // should never happen - condition checked previously
  end;

  procedure DoCompiles(ThisJawsRec: TJawsRecord);
  var
    I: Integer;
  begin
    if not assigned(JAWSAPI) then
      JAWSAPI := CoJawsApi.Create;
    for I := 0 to CompileCommands.Count - 1 do
    begin
      ExecuteAndWait('"' + ThisJawsRec.Compiler + '"', CompileCommands[I]);
      JAWSAPI.StopSpeech;
    end;
    ReloadConfiguration;
  end;

begin
  Result := FALSE;
  for ArryCnt := Low(JawsRecord) to High(JawsRecord) do
  begin
    AddToLog('Start version : ' + FloatToStr(JawsRecord[ArryCnt].Version), 2);
    ScriptFileChanges := FALSE;
    if JAWSManager.RequiredFilesFound then
    begin
      FJAWSFileError := '';
      CompileCommands := TStringList.Create;
      try
        EnsureJAWSScriptsAreUpToDate(JawsRecord[ArryCnt]);
        AddToLog('Compile', 3);
        if CompileCommands.Count > 0 then
          DoCompiles(JawsRecord[ArryCnt]);
      finally
        CompileCommands.Free;
      end;
      if FileErrorExists then
        ShowError(JAWS_ERROR_FILE_IO, [FJAWSFileError])
      else if JAWSTalking2CurrentUser then
      begin
        EnsureWindow;
        AddToLog('Launching master app', 3);
        LaunchMasterApplication;
        if ScriptFileChanges then
        begin
          FMainForm.ConfigReloadNeeded;
        end;
        Result := TRUE;
      end;
    end;
  end;
end;

class function TJAWSManager.IsRunning(HighVersion, LowVersion: Word): BOOL;

  function ComponentVersionSupported: boolean;
  var
    SupportedHighVersion, SupportedLowVersion: Integer;
    FileName, newVersion, convertedVersion, currentVersion: string;
    addr: pointer;

  begin
    addr := @TJAWSManager.IsRunning;
    FileName := GetDLLFileName(addr);
    currentVersion := FileVersionValue(FileName, FILE_VER_FILEVERSION);
    VersionStringSplit(currentVersion, SupportedHighVersion,
      SupportedLowVersion);
    Result := FALSE;
    if (HighVersion < SupportedHighVersion) then
      Result := TRUE
    else if (HighVersion = SupportedHighVersion) and
      (LowVersion <= SupportedLowVersion) then
      Result := TRUE;
    if not Result then
    begin
      newVersion := IntToStr(HighVersion) + '.' + IntToStr(LowVersion);
      convertedVersion := IntToStr(SupportedHighVersion) + '.' +
        IntToStr(SupportedLowVersion);
      ShowError(DLL_ERROR_VERSION, [newVersion, convertedVersion]);
    end;
    AddToLog('DLL Version (' + currentVersion + ') Supported: ' +
      BoolToStr(Result), 3);
  end;

begin
  Result := (GetJAWSWindow <> 0);
  if Result then
    Result := ComponentVersionSupported;
  if Result then
    Result := JAWSVersionOK;
  if Result then
  begin
    EnsureManager;
    with JAWSManager do
      Result := RequiredFilesFound;
  end;
end;

class function TJAWSManager.JAWSTalking2CurrentUser: boolean;
var
  CurrentUserPath: string;
  WhatJAWSThinks: string;

  { -------------------------------------------------------------------------------
    Procedure:   GetProcessUserAndDomain
    Author:      VHAISPBELLC
    DateTime:    2013.08.27
    Arguments:   Pid: DWORD
    Result:      string
    Description: Gathers the username that ran the process
    ------------------------------------------------------------------------------- }
  function GetProcessUserAndDomain(Pid: DWORD): string;
  const
    FlgFoward = $00000020;
  var
    WbemLocator, WMIService, WbemObject, WbemObjectSet: OLEVariant;
    UserName, DomainName: OLEVariant;
    ProcessEnum: IEnumvariant;
    iValue: LongWord;
    QueryToRun: String;
  begin
    QueryToRun := 'SELECT * FROM Win32_Process where Handle=' + IntToStr(Pid);
    WbemLocator := CreateOleObject('WbemScripting.SWbemLocator');
    WMIService := WbemLocator.ConnectServer('localhost', 'root\CIMV2', '', '');
    WbemObjectSet := WMIService.ExecQuery(QueryToRun, 'WQL', FlgFoward);
    ProcessEnum := IUnknown(WbemObjectSet._NewEnum) as IEnumvariant;
    if ProcessEnum.Next(1, WbemObject, iValue) = 0 then
    begin
      WbemObject.GetOwner(UserName, DomainName);
      Result := DomainName + '/' + UserName;
    end
    else
      Result := '';
  end;

  procedure Fix(var path: string);
  var
    idx: Integer;
  begin
    idx := pos(APP_DATA, LowerCase(path));
    if idx > 0 then
      path := LeftStr(path, idx - 1);
    idx := length(path);
    while (idx > 0) and (path[idx] <> '\') do
      dec(idx);
    delete(path, 1, idx);
  end;

  function UserProblemExists: boolean;
  var
    JAWSWindow: HWND;
    pPid: DWORD;
  begin
    JAWSWindow := GetJAWSWindow;
    pPid := INVALID_HANDLE_VALUE;
    GetWindowThreadProcessId(JAWSWindow, @pPid);

    CurrentUserPath := GetProcessUserAndDomain(GetCurrentProcessId);
    WhatJAWSThinks := GetProcessUserAndDomain(pPid);
    Result := (LowerCase(CurrentUserPath) <> LowerCase(WhatJAWSThinks));
  end;

begin
  if UserProblemExists then
  begin
    ShowError(JAWS_ERROR_USER_PROBLEM);
    Result := FALSE;
  end
  else
    Result := TRUE;
end;

class function TJAWSManager.JAWSVersionOK: boolean;
var
  JFileVersion: string;
  JFile: string;
  I: Integer;
  ErrFound, Ok: boolean;

  function OlderVersionOKIfCOMObjectInstalled: boolean;
  var
    api: IJawsApi;
  begin
    Result := VersionOK(JAWS_REQUIRED_VERSION, JFileVersion);
    if Result then
    begin
      try
        try
          api := CoJawsApi.Create;
        except
          Result := FALSE;
        end;
      finally
        api := nil;
      end;
    end;
  end;

begin
  // VHAISPBELLC 8/27/2013 check for old versions
  ErrFound := FALSE;
  for I := Low(JawsRecord) to High(JawsRecord) do
  begin
    JFile := ExtractFilePath(JawsRecord[I].Compiler) +
      JAWS_APPLICATION_FILENAME;
    if FileExists(JFile) then
    begin
      JFileVersion := FileVersionValue(JFile, FILE_VER_FILEVERSION);
      Ok := VersionOK(JAWS_COM_OBJECT_VERSION, JFileVersion);
      if not Ok then
        Ok := OlderVersionOKIfCOMObjectInstalled;
    end
    else
    begin
      // if file not found, then assume a future version where the exe was moved
      // to a different location
      Ok := TRUE;
    end;
    if not Ok then
    begin
      ErrFound := TRUE;
      break;
    end;
    AddToLog('Jaws Application (' + JFileVersion + ') Supported: ' +
      BoolToStr(Ok), 3);
  end;
  if ErrFound then
    ShowError(JAWS_ERROR_VERSION);

  Result := not ErrFound;

end;

procedure TJAWSManager.KillINIFiles(Sender: TObject);
var
 I: Integer;
begin
  for I := Low(JawsRecord) to High(JawsRecord) do
  begin
  if assigned(JawsRecord[i].FDictionaryFile) then
  begin
    if JawsRecord[i].FDictionaryFileModified then
    begin
      MakeFileWritable(JawsRecord[i].FDictionaryFileName);
      JawsRecord[i].FDictionaryFile.SaveToFile(JawsRecord[i].FDictionaryFileName);
    end;
    FreeAndNil(JawsRecord[i].FDictionaryFile);
  end;

  if assigned(JawsRecord[i].FConfigINIFile) then
  begin
    if JawsRecord[i].FConfigINIFileModified then
    begin
      JawsRecord[i].FConfigINIFile.UpdateFile;
    end;
    FreeAndNil(JawsRecord[i].FConfigINIFile);
  end;

  if assigned(JawsRecord[i].FKeyMapINIFile) then
  begin
    if JawsRecord[i].FKeyMapINIFileModified then
    begin
      JawsRecord[i].FKeyMapINIFile.UpdateFile;
    end;
    FreeAndNil(JawsRecord[i].FKeyMapINIFile);
  end;

  if assigned(JawsRecord[i].FAssignedKeys) then
    FreeAndNil(JawsRecord[i].FAssignedKeys);
  end;
end;

procedure TJAWSManager.LaunchMasterApplication;
begin
  if FileExists(FMasterApp) then
    ShellExecute(0, PChar('open'), PChar(FMasterApp), nil,
      PChar(ExtractFilePath(FMasterApp)), SW_SHOWNA);
end;

procedure TJAWSManager.RegisterCustomBehavior(Before, After: string;
  Action: Integer);

const
  WindowClassesSection = 'WindowClasses';
  MSAAClassesSection = 'MSAAClasses';
  DICT_DELIM: char = char($2E);
  CommonKeysSection = 'Common Keys';
  CustomCommandHelpSection = 'Custom Command Help';
  KeyCommand = 'VA508SendCustomCommand(';
  KeyCommandLen = length(KeyCommand);

var
  modified: boolean;

  procedure Add2INIFile(var INIFile: TINIFile; var FileModified: boolean;
    FileName, SectionName, data, Value: string);
  var
    oldValue: string;

  begin
    if not assigned(INIFile) then
    begin
      MakeFileWritable(FileName);
      INIFile := TINIFile.Create(FileName);
      FileModified := FALSE;
    end;
    oldValue := INIFile.ReadString(SectionName, data, '');
    if oldValue <> Value then
    begin
      INIFile.WriteString(SectionName, data, Value);
      modified := TRUE;
      FileModified := TRUE;
    end;
  end;

  procedure RemoveFromINIFile(var INIFile: TINIFile; var FileModified: boolean;
    FileName, SectionName, data: string);
  var
    oldValue: string;

  begin
    if not assigned(INIFile) then
    begin
      MakeFileWritable(FileName);
      INIFile := TINIFile.Create(FileName);
      FileModified := FALSE;
    end;
    oldValue := INIFile.ReadString(SectionName, data, '');
    if oldValue <> '' then
    begin
      INIFile.DeleteKey(SectionName, data);
      modified := TRUE;
      FileModified := TRUE;
    end;
  end;

  procedure RegisterCustomClassChange;
  Var
   I:Integer;
  begin
    for I := Low(JawsRecord) to High(JawsRecord) do
     Add2INIFile(JawsRecord[i].FConfigINIFile, JawsRecord[i].FConfigINIFileModified, JawsRecord[i].FConfigFile,
      WindowClassesSection, Before, After);
  end;

  procedure RegisterMSAAClassChange;
  Var
   I:Integer;
  begin
    for I := Low(JawsRecord) to High(JawsRecord) do
    Add2INIFile(JawsRecord[i].FConfigINIFile, JawsRecord[i].FConfigINIFileModified, JawsRecord[i].FConfigFile,
      MSAAClassesSection, Before, '1');
  end;

  procedure RegisterCustomKeyMapping;
  Var
   I:Integer;
  begin
    for I := Low(JawsRecord) to High(JawsRecord) do
    begin
    Add2INIFile(JawsRecord[i].FKeyMapINIFile, JawsRecord[i].FKeyMapINIFileModified, JawsRecord[i].FKeyMapFile,
      CommonKeysSection, Before, KeyCommand + After + ')');
    if not assigned(JawsRecord[i].FAssignedKeys) then
      JawsRecord[i].FAssignedKeys := TStringList.Create;
    JawsRecord[i].FAssignedKeys.Add(Before);
    end;
  end;

  procedure RegisterCustomKeyDescription;
  Var
   I:Integer;
  begin
    for I := Low(JawsRecord) to High(JawsRecord) do
    Add2INIFile(JawsRecord[i].FConfigINIFile, JawsRecord[i].FConfigINIFileModified, JawsRecord[i].FConfigFile,
      CustomCommandHelpSection, Before, After);
  end;

  procedure DecodeLine(line: string; var before1, after1: string);
  var
    I, j, len: Integer;
  begin
    before1 := '';
    after1 := '';
    len := length(line);
    if (len < 2) or (line[1] <> DICT_DELIM) then
      exit;
    I := 2;
    while (I < len) and (line[I] <> DICT_DELIM) do
      inc(I);
    before1 := copy(line, 2, I - 2);
    j := I + 1;
    while (j <= len) and (line[j] <> DICT_DELIM) do
      inc(j);
    after1 := copy(line, I + 1, j - I - 1);
  end;

  procedure RegisterCustomDictionaryChange;
  var
    I, idx, X: Integer;
    line, before1, after1: string;
    Add: boolean;
  begin
    for X := Low(JawsRecord) to High(JawsRecord) do
    begin
    if not assigned(JawsRecord[X].FDictionaryFile) then
    begin
      JawsRecord[X].FDictionaryFile := TStringList.Create;
      JawsRecord[X].FDictionaryFileModified := FALSE;
      if FileExists(JawsRecord[X].FDictionaryFileName) then
        JawsRecord[X].FDictionaryFile.LoadFromFile(JawsRecord[X].FDictionaryFileName);
    end;

    Add := TRUE;
    idx := -1;
    for I := 0 to JawsRecord[X].FDictionaryFile.Count - 1 do
    begin
      line := JawsRecord[X].FDictionaryFile[I];
      DecodeLine(line, before1, after1);
      if (before1 = Before) then
      begin
        idx := I;
        if after1 = After then
          Add := FALSE;
        break;
      end;
    end;
    if Add then
    begin
      line := DICT_DELIM + Before + DICT_DELIM + After + DICT_DELIM;
      if idx < 0 then
        JawsRecord[X].FDictionaryFile.Add(line)
      else
        JawsRecord[X].FDictionaryFile[idx] := line;
      modified := TRUE;
      JawsRecord[X].FDictionaryFileModified := TRUE;
    end;
    end;
  end;

  procedure RemoveComponentClass;
  Var
   I:Integer;
  begin
    for I := Low(JawsRecord) to High(JawsRecord) do
    RemoveFromINIFile(JawsRecord[i].FConfigINIFile, JawsRecord[i].FConfigINIFileModified, JawsRecord[i].FConfigFile,
      WindowClassesSection, Before);
  end;

  procedure RemoveMSAAClass;
   Var
   I:Integer;
  begin
    for I := Low(JawsRecord) to High(JawsRecord) do
    RemoveFromINIFile(JawsRecord[i].FConfigINIFile, JawsRecord[i].FConfigINIFileModified, JawsRecord[i].FConfigFile,
      MSAAClassesSection, Before);
  end;

  procedure PurgeKeyMappings;
  var
    I, X: Integer;
    name, Value: string;
    keys: TStringList;
    delete: boolean;
  begin
    for X := Low(JawsRecord) to High(JawsRecord) do
    begin
    if not assigned(JawsRecord[X].FKeyMapINIFile) then
    begin
      MakeFileWritable(JawsRecord[X].FKeyMapFile);
      JawsRecord[X].FKeyMapINIFile := TINIFile.Create(JawsRecord[X].FKeyMapFile);
      JawsRecord[X].FKeyMapINIFileModified := FALSE;
    end;
    keys := TStringList.Create;
    try
      JawsRecord[X].FKeyMapINIFile.ReadSectionValues(CommonKeysSection, keys);
      for I := keys.Count - 1 downto 0 do
      begin
        Value := copy(keys.ValueFromIndex[I], 1, KeyCommandLen);
        if Value = KeyCommand then
        begin
          name := keys.Names[I];
          delete := (not assigned(JawsRecord[X].FAssignedKeys));
          if not delete then
            delete := (JawsRecord[X].FAssignedKeys.IndexOf(name) < 0);
          if delete then
          begin
            JawsRecord[X].FKeyMapINIFile.DeleteKey(CommonKeysSection, name);
            JawsRecord[X].FKeyMapINIFileModified := TRUE;
            modified := TRUE;
          end;
        end;
      end;
    finally
      keys.Free;
    end;
    end;
  end;

begin
  { TODO : check file io errors when updating config files }
  modified := FALSE;
  case Action of
    BEHAVIOR_ADD_DICTIONARY_CHANGE:
      RegisterCustomDictionaryChange;
    BEHAVIOR_ADD_COMPONENT_CLASS:
      RegisterCustomClassChange;
    BEHAVIOR_ADD_COMPONENT_MSAA:
      RegisterMSAAClassChange;
    BEHAVIOR_ADD_CUSTOM_KEY_MAPPING:
      RegisterCustomKeyMapping;
    BEHAVIOR_ADD_CUSTOM_KEY_DESCRIPTION:
      RegisterCustomKeyDescription;
    BEHAVIOR_REMOVE_COMPONENT_CLASS:
      RemoveComponentClass;
    BEHAVIOR_REMOVE_COMPONENT_MSAA:
      RemoveMSAAClass;
    BEHAVIOR_PURGE_UNREGISTERED_KEY_MAPPINGS:
      PurgeKeyMappings;
  end;
  if modified and assigned(FMainForm) then
  begin
    FMainForm.ResetINITimer(KillINIFiles);
    FMainForm.ConfigReloadNeeded;
  end;
end;

procedure TJAWSManager.ReloadConfiguration;
begin
  if not assigned(JAWSAPI) then
    JAWSAPI := CoJawsApi.Create;
  JAWSAPI.RunFunction('ReloadAllConfigs');
end;

procedure TJAWSManager.SendComponentData(WindowHandle: HWND;
  DataStatus: LongInt; Caption, Value, data, ControlType, State, Instructions,
  ItemInstructions: PChar);

  procedure SendRequestResponse;
  begin
    FMainForm.WriteData(VA508_REG_COMPONENT_CAPTION, Caption);
    FMainForm.WriteData(VA508_REG_COMPONENT_VALUE, Value);
    FMainForm.WriteData(VA508_REG_COMPONENT_CONTROL_TYPE, ControlType);
    FMainForm.WriteData(VA508_REG_COMPONENT_STATE, State);
    FMainForm.WriteData(VA508_REG_COMPONENT_INSTRUCTIONS, Instructions);
    FMainForm.WriteData(VA508_REG_COMPONENT_ITEM_INSTRUCTIONS,
      ItemInstructions);
    FMainForm.WriteData(VA508_REG_COMPONENT_DATA_STATUS, IntToStr(DataStatus));
    FMainForm.PostData;
  end;

  procedure SendChangeEvent;
  var
    Event: WideString;
  begin
    Event := 'VA508ChangeEvent(' + IntToStr(WindowHandle) + ',' +
      IntToStr(DataStatus) + ',"' + StrPas(Caption) + '","' + StrPas(Value) +
      '","' + StrPas(ControlType) + '","' + StrPas(State) + '","' +
      StrPas(Instructions) + '","' + StrPas(ItemInstructions) + '"';
    if not assigned(JAWSAPI) then
      JAWSAPI := CoJawsApi.Create;
    JAWSAPI.RunFunction(Event)
  end;

begin
  if (data <> nil) and (length(data) > 0) then
  begin
    Value := data;
    DataStatus := DataStatus AND DATA_MASK_DATA;
    DataStatus := DataStatus OR DATA_VALUE;
  end;
  if (DataStatus and DATA_CHANGE_EVENT) <> 0 then
  begin
    DataStatus := DataStatus AND DATA_MASK_CHANGE_EVENT;
    SendChangeEvent;
  end
  else
    SendRequestResponse;
end;

const
  MAX_REG_CHARS = 125;
  // When Jaws reads over 126 chars it returns a blank string
  MORE_STRINGS = '+';
  LAST_STRING = '-';
  MAX_COUNT_KEY = 'Max';

class procedure TJAWSManager.ShowError(ErrorNumber: Integer);
begin
  ShowError(ErrorNumber, []);
end;

class procedure TJAWSManager.ShowError(ErrorNumber: Integer;
  data: array of const);
var
  error: string;

begin
  if not JAWSErrorsShown[ErrorNumber] then
  begin
    error := JAWSErrorMessage[ErrorNumber];
    if length(data) > 0 then
      error := Format(error, data);
    JAWSErrorsShown[ErrorNumber] := TRUE;
    MessageBox(0, PChar(error), 'JAWS Accessibility Component Error',
      MB_OK or MB_ICONERROR or MB_TASKMODAL or MB_TOPMOST);
  end;
end;

procedure TJAWSManager.ShutDown;
begin
  if FWasShutdown then
    exit;
  if assigned(JAWSAPI) then
  begin
    try
      JAWSAPI := nil; // causes access violation
    except
    end;
  end;
  KillINIFiles(nil);
  if assigned(FMainForm) then
    FreeAndNil(FMainForm);
  FWasShutdown := TRUE;
end;

procedure TJAWSManager.SpeakText(Text: PChar);
begin
  if not assigned(JAWSAPI) then
    JAWSAPI := CoJawsApi.Create;
  JAWSAPI.SayString(Text, FALSE);
end;

initialization

CoInitializeEx(nil, COINIT_APARTMENTTHREADED);

finalization

ShutDown;
CoUninitialize;

end.
